<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Users - ReviewPR</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Inter', Arial, sans-serif; background: #f8f9fa; }
        .container { max-width: 1200px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2rem; border-radius: 12px; margin-bottom: 2rem; }
        .card { background: white; border-radius: 12px; padding: 2rem; margin-bottom: 2rem; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        .form-group { margin-bottom: 1.5rem; }
        .form-label { display: block; font-weight: 600; margin-bottom: 0.5rem; color: #374151; }
        .form-control { width: 100%; padding: 0.75rem; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1rem; }
        .form-control:focus { outline: none; border-color: #667eea; }
        .btn { padding: 0.75rem 1.5rem; border: none; border-radius: 8px; font-weight: 600; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn-primary { background: #667eea; color: white; }
        .btn-success { background: #10b981; color: white; }
        .btn-danger { background: #ef4444; color: white; }
        .btn-secondary { background: #6b7280; color: white; }
        .btn-warning { background: #f59e0b; color: white; }
        .btn:hover { opacity: 0.9; transform: translateY(-1px); }
        .grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 1.5rem; }
        .user-card { background: white; border-radius: 12px; padding: 1.5rem; box-shadow: 0 2px 4px rgba(0,0,0,0.1); border: 1px solid #e5e7eb; }
        .alert { padding: 1rem; border-radius: 8px; margin-bottom: 1rem; }
        .alert-success { background: #d1fae5; color: #065f46; border: 1px solid #a7f3d0; }
        .alert-error { background: #fee2e2; color: #991b1b; border: 1px solid #fecaca; }
        .nav-links { margin-bottom: 2rem; }
        .nav-links a { margin-right: 1rem; }
        .role-badge { padding: 0.25rem 0.75rem; border-radius: 20px; font-size: 0.875rem; font-weight: 600; }
        .role-admin { background: #fee2e2; color: #991b1b; }
        .role-staff { background: #dbeafe; color: #1e40af; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>👥 User Management System</h1>
            <p>Add, edit, and manage system users</p>
        </div>

        <div class="nav-links">
            <a href="admin_dashboard.php" class="btn btn-secondary">← Back to Dashboard</a>
        </div>

        <?php
        require_once 'config.php';

        // Handle form submissions
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $conn = getSecureConnection();

            if (isset($_POST['action'])) {
                switch ($_POST['action']) {
                    case 'add':
                        $username = trim($_POST['username']);
                        $email = trim($_POST['email']);
                        $password = $_POST['password'];
                        $role = $_POST['role'];

                        // Validate input
                        if (empty($username) || empty($email) || empty($password) || empty($role)) {
                            echo '<div class="alert alert-error">❌ All fields are required!</div>';
                        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                            echo '<div class="alert alert-error">❌ Invalid email format!</div>';
                        } else {
                            // Check if username or email already exists
                            $check = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
                            $check->bind_param("ss", $username, $email);
                            $check->execute();
                            if ($check->get_result()->num_rows > 0) {
                                echo '<div class="alert alert-error">❌ Username or email already exists!</div>';
                            } else {
                                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                                $stmt = $conn->prepare("INSERT INTO users (username, email, password, role) VALUES (?, ?, ?, ?)");
                                $stmt->bind_param("ssss", $username, $email, $hashed_password, $role);

                                if ($stmt->execute()) {
                                    echo '<div class="alert alert-success">✅ User added successfully!</div>';
                                } else {
                                    echo '<div class="alert alert-error">❌ Error adding user: ' . $stmt->error . '</div>';
                                }
                                $stmt->close();
                            }
                            $check->close();
                        }
                        break;

                    case 'update':
                        $id = $_POST['id'];
                        $username = trim($_POST['username']);
                        $email = trim($_POST['email']);
                        $role = $_POST['role'];
                        $is_active = isset($_POST['is_active']) ? 1 : 0;

                        // Validate input
                        if (empty($username) || empty($email) || empty($role)) {
                            echo '<div class="alert alert-error">❌ All fields are required!</div>';
                        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                            echo '<div class="alert alert-error">❌ Invalid email format!</div>';
                        } else {
                            // Check if username or email already exists (excluding current user)
                            $check = $conn->prepare("SELECT id FROM users WHERE (username = ? OR email = ?) AND id != ?");
                            $check->bind_param("ssi", $username, $email, $id);
                            $check->execute();
                            if ($check->get_result()->num_rows > 0) {
                                echo '<div class="alert alert-error">❌ Username or email already exists!</div>';
                            } else {
                                $stmt = $conn->prepare("UPDATE users SET username=?, email=?, role=?, is_active=? WHERE id=?");
                                $stmt->bind_param("sssii", $username, $email, $role, $is_active, $id);

                                if ($stmt->execute()) {
                                    echo '<div class="alert alert-success">✅ User updated successfully!</div>';
                                } else {
                                    echo '<div class="alert alert-error">❌ Error updating user: ' . $stmt->error . '</div>';
                                }
                                $stmt->close();
                            }
                            $check->close();
                        }
                        break;

                    case 'delete':
                        $id = $_POST['id'];
                        // Prevent deleting the current admin
                        if ($id == $_SESSION['admin_id']) {
                            echo '<div class="alert alert-error">❌ Cannot delete your own account!</div>';
                        } else {
                            $stmt = $conn->prepare("DELETE FROM users WHERE id=?");
                            $stmt->bind_param("i", $id);

                            if ($stmt->execute()) {
                                echo '<div class="alert alert-success">✅ User deleted successfully!</div>';
                            } else {
                                echo '<div class="alert alert-error">❌ Error deleting user: ' . $stmt->error . '</div>';
                            }
                            $stmt->close();
                        }
                        break;

                    case 'change_password':
                        $current_password = $_POST['current_password'];
                        $new_password = $_POST['new_password'];
                        $confirm_password = $_POST['confirm_password'];

                        if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
                            echo '<div class="alert alert-error">❌ All password fields are required!</div>';
                        } elseif ($new_password !== $confirm_password) {
                            echo '<div class="alert alert-error">❌ New passwords do not match!</div>';
                        } elseif (strlen($new_password) < 6) {
                            echo '<div class="alert alert-error">❌ New password must be at least 6 characters!</div>';
                        } else {
                            // Verify current password
                            $stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
                            $stmt->bind_param("i", $_SESSION['admin_id']);
                            $stmt->execute();
                            $result = $stmt->get_result();
                            if ($result->num_rows === 1) {
                                $user = $result->fetch_assoc();
                                if (password_verify($current_password, $user['password'])) {
                                    // Update password
                                    $hashed_new_password = password_hash($new_password, PASSWORD_DEFAULT);
                                    $update_stmt = $conn->prepare("UPDATE users SET password = ? WHERE id = ?");
                                    $update_stmt->bind_param("si", $hashed_new_password, $_SESSION['admin_id']);

                                    if ($update_stmt->execute()) {
                                        echo '<div class="alert alert-success">✅ Password changed successfully!</div>';
                                    } else {
                                        echo '<div class="alert alert-error">❌ Error updating password: ' . $update_stmt->error . '</div>';
                                    }
                                    $update_stmt->close();
                                } else {
                                    echo '<div class="alert alert-error">❌ Current password is incorrect!</div>';
                                }
                            } else {
                                echo '<div class="alert alert-error">❌ User not found!</div>';
                            }
                            $stmt->close();
                        }
                        break;

                    case 'change_username':
                        $new_username = trim($_POST['new_username']);
                        $password = $_POST['password'];

                        if (empty($new_username) || empty($password)) {
                            echo '<div class="alert alert-error">❌ Username and password are required!</div>';
                        } else {
                            // Verify password
                            $stmt = $conn->prepare("SELECT password FROM users WHERE id = ?");
                            $stmt->bind_param("i", $_SESSION['admin_id']);
                            $stmt->execute();
                            $result = $stmt->get_result();
                            if ($result->num_rows === 1) {
                                $user = $result->fetch_assoc();
                                if (password_verify($password, $user['password'])) {
                                    // Check if new username already exists
                                    $check = $conn->prepare("SELECT id FROM users WHERE username = ? AND id != ?");
                                    $check->bind_param("si", $new_username, $_SESSION['admin_id']);
                                    $check->execute();
                                    if ($check->get_result()->num_rows > 0) {
                                        echo '<div class="alert alert-error">❌ Username already exists!</div>';
                                    } else {
                                        // Update username
                                        $update_stmt = $conn->prepare("UPDATE users SET username = ? WHERE id = ?");
                                        $update_stmt->bind_param("si", $new_username, $_SESSION['admin_id']);

                                        if ($update_stmt->execute()) {
                                            // Update session
                                            $_SESSION['admin_username'] = $new_username;
                                            echo '<div class="alert alert-success">✅ Username changed successfully!</div>';
                                        } else {
                                            echo '<div class="alert alert-error">❌ Error updating username: ' . $update_stmt->error . '</div>';
                                        }
                                        $update_stmt->close();
                                    }
                                    $check->close();
                                } else {
                                    echo '<div class="alert alert-error">❌ Password is incorrect!</div>';
                                }
                            } else {
                                echo '<div class="alert alert-error">❌ User not found!</div>';
                            }
                            $stmt->close();
                        }
                        break;
                }
            }

            $conn->close();
        }
        ?>

        <!-- Change Own Credentials -->
        <div class="card">
            <h2>🔐 Change Your Credentials</h2>

            <!-- Change Username Form -->
            <div style="margin-bottom: 2rem; padding-bottom: 2rem; border-bottom: 1px solid #e5e7eb;">
                <h3>Change Username</h3>
                <form method="POST" style="margin-top: 1rem;">
                    <input type="hidden" name="action" value="change_username">

                    <div class="grid">
                        <div class="form-group">
                            <label class="form-label">New Username</label>
                            <input type="text" name="new_username" class="form-control" required placeholder="Enter new username">
                        </div>

                        <div class="form-group">
                            <label class="form-label">Current Password</label>
                            <input type="password" name="password" class="form-control" required placeholder="Enter current password">
                        </div>
                    </div>

                    <button type="submit" class="btn btn-primary">💾 Change Username</button>
                </form>
            </div>

            <!-- Change Password Form -->
            <h3>Change Password</h3>
            <form method="POST" style="margin-top: 1rem;">
                <input type="hidden" name="action" value="change_password">

                <div class="form-group">
                    <label class="form-label">Current Password</label>
                    <input type="password" name="current_password" class="form-control" required placeholder="Enter current password">
                </div>

                <div class="grid">
                    <div class="form-group">
                        <label class="form-label">New Password</label>
                        <input type="password" name="new_password" class="form-control" required placeholder="Enter new password" minlength="6">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Confirm New Password</label>
                        <input type="password" name="confirm_password" class="form-control" required placeholder="Confirm new password" minlength="6">
                    </div>
                </div>

                <button type="submit" class="btn btn-warning">🔒 Change Password</button>
            </form>
        </div>

        <!-- Add New User Form -->
        <div class="card">
            <h2>➕ Add New User</h2>
            <form method="POST">
                <input type="hidden" name="action" value="add">

                <div class="grid">
                    <div class="form-group">
                        <label class="form-label">Username *</label>
                        <input type="text" name="username" class="form-control" required placeholder="Enter username">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Email *</label>
                        <input type="email" name="email" class="form-control" required placeholder="Enter email">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Password *</label>
                        <input type="password" name="password" class="form-control" required placeholder="Enter password" minlength="6">
                    </div>

                    <div class="form-group">
                        <label class="form-label">Role *</label>
                        <select name="role" class="form-control" required>
                            <option value="">Select Role</option>
                            <option value="admin">Admin</option>
                            <option value="staff">Staff</option>
                        </select>
                    </div>
                </div>

                <button type="submit" class="btn btn-success">➕ Add User</button>
            </form>
        </div>

        <!-- Existing Users -->
        <div class="card">
            <h2>📋 Existing Users</h2>

            <?php
            try {
                $conn = getSecureConnection();

                $result = $conn->query("SELECT id, username, email, role, is_active, created_at FROM users ORDER BY id ASC");

                if ($result && $result->num_rows > 0) {
                    echo '<div class="grid">';

                    while($user = $result->fetch_assoc()) {
                        $roleClass = $user['role'] === 'admin' ? 'role-admin' : 'role-staff';
                        $statusText = $user['is_active'] ? 'Active' : 'Inactive';
                        $statusClass = $user['is_active'] ? 'role-staff' : 'role-admin';

                        echo '<div class="user-card">';
                        echo '<h3>' . htmlspecialchars($user['username']) . '</h3>';
                        echo '<p><strong>Email:</strong> ' . htmlspecialchars($user['email']) . '</p>';
                        echo '<p><strong>Role:</strong> <span class="role-badge ' . $roleClass . '">' . ucfirst($user['role']) . '</span></p>';
                        echo '<p><strong>Status:</strong> <span class="role-badge ' . $statusClass . '">' . $statusText . '</span></p>';
                        echo '<p><strong>Created:</strong> ' . date('M d, Y', strtotime($user['created_at'])) . '</p>';

                        if ($user['id'] != $_SESSION['admin_id']) {
                            echo '<div style="margin-top: 1rem;">';
                            echo '<button onclick="editUser(' . $user['id'] . ')" class="btn btn-primary">✏️ Edit</button> ';
                            echo '<form method="POST" style="display: inline;" onsubmit="return confirm(\'Are you sure you want to delete this user?\')">';
                            echo '<input type="hidden" name="action" value="delete">';
                            echo '<input type="hidden" name="id" value="' . $user['id'] . '">';
                            echo '<button type="submit" class="btn btn-danger">🗑️ Delete</button>';
                            echo '</form>';
                            echo '</div>';
                        } else {
                            echo '<div style="margin-top: 1rem;">';
                            echo '<span class="btn btn-secondary" style="opacity: 0.6;">👤 Current User</span>';
                            echo '</div>';
                        }

                        echo '</div>';
                    }

                    echo '</div>';
                } else {
                    echo '<p>No users found.</p>';
                }

                $conn->close();
            } catch (Exception $e) {
                echo '<div class="alert alert-error">❌ Error loading users: ' . $e->getMessage() . '</div>';
            }
            ?>
        </div>
    </div>

    <!-- Edit User Modal -->
    <div id="editModal" class="modal" style="display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; overflow: auto; background-color: rgba(0,0,0,0.4);">
        <div class="modal-content" style="background-color: #fefefe; margin: 5% auto; padding: 20px; border: 1px solid #888; width: 80%; max-width: 600px; border-radius: 12px;">
            <span class="close" onclick="closeModal()" style="color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer;">&times;</span>
            <h2>✏️ Edit User</h2>
            <form method="POST" id="editForm">
                <input type="hidden" name="action" value="update">
                <input type="hidden" name="id" id="edit-id">

                <div class="form-group">
                    <label class="form-label">Username *</label>
                    <input type="text" name="username" id="edit-username" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label">Email *</label>
                    <input type="email" name="email" id="edit-email" class="form-control" required>
                </div>

                <div class="form-group">
                    <label class="form-label">Role *</label>
                    <select name="role" id="edit-role" class="form-control" required>
                        <option value="">Select Role</option>
                        <option value="admin">Admin</option>
                        <option value="staff">Staff</option>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">Status</label>
                    <div style="margin-top: 0.5rem;">
                        <input type="checkbox" name="is_active" id="edit-is-active" value="1" style="margin-right: 0.5rem;">
                        <label for="edit-is-active" style="display: inline;">Active</label>
                    </div>
                </div>

                <button type="submit" class="btn btn-primary">💾 Update User</button>
                <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            </form>
        </div>
    </div>

    <script>
        function editUser(id) {
            // Get user data via AJAX
            fetch('get_user.php?id=' + id)
                .then(response => response.json())
                .then(user => {
                    if (user) {
                        // Populate modal with user data
                        document.getElementById('edit-id').value = user.id;
                        document.getElementById('edit-username').value = user.username;
                        document.getElementById('edit-email').value = user.email;
                        document.getElementById('edit-role').value = user.role;
                        document.getElementById('edit-is-active').checked = user.is_active == 1;

                        // Show modal
                        document.getElementById('editModal').style.display = 'block';
                    } else {
                        alert('User not found!');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading user data!');
                });
        }

        function closeModal() {
            document.getElementById('editModal').style.display = 'none';
        }

        // Close modal when clicking outside of it
        window.onclick = function(event) {
            const modal = document.getElementById('editModal');
            if (event.target == modal) {
                modal.style.display = 'none';
            }
        }

        // Handle form submission for editing
        document.getElementById('editForm').addEventListener('submit', function(e) {
            e.preventDefault(); // Prevent default form submission

            const formData = new FormData(this);

            fetch('', {
                method: 'POST',
                body: formData
            })
            .then(response => response.text())
            .then(result => {
                if (result.includes('User updated successfully')) {
                    alert('User updated successfully!');
                    closeModal();
                    location.reload(); // Refresh page to show updated data
                } else {
                    alert('Error updating user: ' + result);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Error updating user!');
            });
        });
    </script>
</body>
</html>
